<?php
/**
 * Classe pour gérer l'interface d'administration
 */

if (!defined('ABSPATH')) {
    exit;
}

class Leonix_Admin {
    
    /**
     * Constructeur
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    /**
     * Ajouter le menu d'administration
     */
    public function add_admin_menu() {
        add_options_page(
            __('Leonix AI Widget', 'leonix-ai-widget'),
            __('Leonix AI', 'leonix-ai-widget'),
            'manage_options',
            'leonix-ai-widget',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Enregistrer les paramètres
     */
    public function register_settings() {
        register_setting('leonix_widget_settings', 'leonix_agent_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));
        
        register_setting('leonix_widget_settings', 'leonix_api_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => 'https://app.leonix.ai/api'
        ));
        
        register_setting('leonix_widget_settings', 'leonix_widget_enabled', array(
            'type' => 'boolean',
            'default' => true
        ));
        
        register_setting('leonix_widget_settings', 'leonix_profile_image', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        
        register_setting('leonix_widget_settings', 'leonix_widget_position', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'right'
        ));
        
        add_settings_section(
            'leonix_widget_main_section',
            __('Configuration du Widget', 'leonix-ai-widget'),
            array($this, 'render_section_description'),
            'leonix_widget_settings'
        );
        
        add_settings_field(
            'leonix_agent_id',
            __('Agent ID', 'leonix-ai-widget'),
            array($this, 'render_agent_id_field'),
            'leonix_widget_settings',
            'leonix_widget_main_section'
        );
        
        add_settings_field(
            'leonix_api_url',
            __('URL de l\'API', 'leonix-ai-widget'),
            array($this, 'render_api_url_field'),
            'leonix_widget_settings',
            'leonix_widget_main_section'
        );
        
        add_settings_field(
            'leonix_widget_enabled',
            __('Activer le widget', 'leonix-ai-widget'),
            array($this, 'render_enabled_field'),
            'leonix_widget_settings',
            'leonix_widget_main_section'
        );
        
        add_settings_field(
            'leonix_profile_image',
            __('Image de profil', 'leonix-ai-widget'),
            array($this, 'render_profile_image_field'),
            'leonix_widget_settings',
            'leonix_widget_main_section'
        );
        
        add_settings_field(
            'leonix_widget_position',
            __('Position du widget', 'leonix-ai-widget'),
            array($this, 'render_position_field'),
            'leonix_widget_settings',
            'leonix_widget_main_section'
        );
    }
    
    /**
     * Afficher la description de la section
     */
    public function render_section_description() {
        echo '<p>' . __('Configurez votre widget Leonix AI pour l\'intégrer à votre site WordPress.', 'leonix-ai-widget') . '</p>';
    }
    
    /**
     * Afficher le champ Agent ID
     */
    public function render_agent_id_field() {
        $value = get_option('leonix_agent_id', '');
        ?>
        <input type="text" 
               id="leonix_agent_id" 
               name="leonix_agent_id" 
               value="<?php echo esc_attr($value); ?>" 
               class="regular-text"
               placeholder="<?php echo esc_attr__('Entrez votre Agent ID', 'leonix-ai-widget'); ?>" />
        <p class="description">
            <?php echo __('Copiez votre Agent ID depuis votre tableau de bord Leonix.', 'leonix-ai-widget'); ?>
        </p>
        <?php
    }
    
    /**
     * Afficher le champ URL de l'API
     */
    public function render_api_url_field() {
        $value = get_option('leonix_api_url', 'https://app.leonix.ai/api');
        ?>
        <input type="url" 
               id="leonix_api_url" 
               name="leonix_api_url" 
               value="<?php echo esc_url($value); ?>" 
               class="regular-text" />
        <p class="description">
            <?php echo __('URL de base de l\'API Leonix (par défaut: https://app.leonix.ai/api)', 'leonix-ai-widget'); ?>
        </p>
        <?php
    }
    
    /**
     * Afficher le champ Activer le widget
     */
    public function render_enabled_field() {
        $value = get_option('leonix_widget_enabled', true);
        ?>
        <label>
            <input type="checkbox" 
                   id="leonix_widget_enabled" 
                   name="leonix_widget_enabled" 
                   value="1" 
                   <?php checked($value, true); ?> />
            <?php echo __('Afficher le widget sur le site', 'leonix-ai-widget'); ?>
        </label>
        <?php
    }
    
    /**
     * Afficher le champ Image de profil
     */
    public function render_profile_image_field() {
        $value = get_option('leonix_profile_image', '');
        ?>
        <div>
            <input type="hidden" 
                   id="leonix_profile_image" 
                   name="leonix_profile_image" 
                   value="<?php echo esc_url($value); ?>" />
            <div id="leonix_profile_image_preview" style="margin-bottom: 10px;">
                <?php if ($value): ?>
                    <img src="<?php echo esc_url($value); ?>" 
                         alt="Profil" 
                         style="width: 100px; height: 100px; object-fit: cover; border-radius: 8px; border: 2px solid #ddd;" />
                <?php else: ?>
                    <div style="width: 100px; height: 100px; background: #f0f0f0; border-radius: 8px; border: 2px solid #ddd; display: flex; align-items: center; justify-content: center; color: #666;">
                        Aucune image
                    </div>
                <?php endif; ?>
            </div>
            <button type="button" 
                    class="button" 
                    id="leonix_upload_image_button">
                <?php echo __('Choisir une image', 'leonix-ai-widget'); ?>
            </button>
            <button type="button" 
                    class="button" 
                    id="leonix_remove_image_button" 
                    style="<?php echo empty($value) ? 'display:none;' : ''; ?>">
                <?php echo __('Supprimer', 'leonix-ai-widget'); ?>
            </button>
            <p class="description">
                <?php echo __('Téléchargez une image carrée pour remplacer l\'icône par défaut. Taille recommandée: 200x200px ou plus.', 'leonix-ai-widget'); ?>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            var mediaUploader;
            
            $('#leonix_upload_image_button').on('click', function(e) {
                e.preventDefault();
                
                if (mediaUploader) {
                    mediaUploader.open();
                    return;
                }
                
                mediaUploader = wp.media({
                    title: '<?php echo esc_js(__('Choisir une image de profil', 'leonix-ai-widget')); ?>',
                    button: {
                        text: '<?php echo esc_js(__('Utiliser cette image', 'leonix-ai-widget')); ?>'
                    },
                    multiple: false,
                    library: {
                        type: 'image'
                    }
                });
                
                mediaUploader.on('select', function() {
                    var attachment = mediaUploader.state().get('selection').first().toJSON();
                    $('#leonix_profile_image').val(attachment.url);
                    $('#leonix_profile_image_preview').html(
                        '<img src="' + attachment.url + '" alt="Profil" style="width: 100px; height: 100px; object-fit: cover; border-radius: 8px; border: 2px solid #ddd;" />'
                    );
                    $('#leonix_remove_image_button').show();
                });
                
                mediaUploader.open();
            });
            
            $('#leonix_remove_image_button').on('click', function(e) {
                e.preventDefault();
                $('#leonix_profile_image').val('');
                $('#leonix_profile_image_preview').html(
                    '<div style="width: 100px; height: 100px; background: #f0f0f0; border-radius: 8px; border: 2px solid #ddd; display: flex; align-items: center; justify-content: center; color: #666;">Aucune image</div>'
                );
                $(this).hide();
            });
        });
        </script>
        <?php
    }
    
    /**
     * Afficher le champ Position du widget
     */
    public function render_position_field() {
        $value = get_option('leonix_widget_position', 'right');
        ?>
        <select id="leonix_widget_position" name="leonix_widget_position">
            <option value="right" <?php selected($value, 'right'); ?>>
                <?php echo __('Droite', 'leonix-ai-widget'); ?>
            </option>
            <option value="left" <?php selected($value, 'left'); ?>>
                <?php echo __('Gauche', 'leonix-ai-widget'); ?>
            </option>
        </select>
        <p class="description">
            <?php echo __('Choisissez la position du widget sur l\'écran.', 'leonix-ai-widget'); ?>
        </p>
        <?php
    }
    
    /**
     * Afficher la page de paramètres
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Afficher les messages de succès
        if (isset($_GET['settings-updated'])) {
            add_settings_error('leonix_widget_messages', 'leonix_widget_message', __('Paramètres enregistrés avec succès.', 'leonix-ai-widget'), 'updated');
        }
        
        settings_errors('leonix_widget_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('leonix_widget_settings');
                do_settings_sections('leonix_widget_settings');
                submit_button(__('Enregistrer les modifications', 'leonix-ai-widget'));
                ?>
            </form>
            
            <div class="leonix-widget-info" style="margin-top: 30px; padding: 20px; background: #f0f0f1; border-left: 4px solid #2271b1;">
                <h2><?php echo __('Instructions', 'leonix-ai-widget'); ?></h2>
                <ol>
                    <li><?php echo __('Connectez-vous à votre tableau de bord Leonix.', 'leonix-ai-widget'); ?></li>
                    <li><?php echo __('Allez dans la section "Widget WordPress" de votre agent.', 'leonix-ai-widget'); ?></li>
                    <li><?php echo __('Copiez votre Agent ID.', 'leonix-ai-widget'); ?></li>
                    <li><?php echo __('Collez l\'Agent ID dans le champ ci-dessus et enregistrez.', 'leonix-ai-widget'); ?></li>
                    <li><?php echo __('Le widget apparaîtra automatiquement sur votre site.', 'leonix-ai-widget'); ?></li>
                </ol>
            </div>
        </div>
        <?php
    }
}

